'use strict';

/* --------------------------------------------------------------
 input_number.js 2025-05-13
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2025 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Widget to add + and - buttons to an input field
 */
gambio.widgets.module('input_number', [], function (data) {

    'use strict';

    // ########## VARIABLE INITIALIZATION ##########

    var $this = $(this),
        separator = null,
        regex = null,
        quantityCheckDelay = 300,
        quantityCheckTimeout = null,
        defaults = {
        // Set the type of the number field. Can be "int" or "float"
        type: 'float',
        // Digits after the locale separator (. or ,)
        digits: 4,
        // Show digits if the are zero
        forceDigits: false,
        // Stepping of the numbers
        stepping: 1,
        // Minimum value of the input field
        min: 0,
        // Maximum value of the input field
        max: null,
        // Set the locale separator (e.g.: . or ,) or set it to "auto" for auto-detection
        separator: 'auto',
        // Initial delay after the mousedown event method gets called again
        delay: 500,
        // Minimum delay that is used for repeating the mousedown event method
        minDelay: 50
    },
        options = $.extend(true, {}, defaults, data),
        module = {};

    // ########## HELPER FUNCTIONS ##########

    /**
     * Helper function that tries to detect the local
     * digits separator
     * @return     {string}        Returns the separator as a string
     * @private
     */
    var _getSeparator = function _getSeparator() {

        if (!options.separator || options.separator === 'auto') {
            var number = 1.1;
            options.separator = number.toLocaleString().substring(1, 2);
            return options.separator;
        }

        return options.separator;
    };

    /**
     * Function to calculate the new value of the input field
     * @param       {object}    e       jQuery event object that gets passed from the event listener below
     * @private
     */
    var _update = function _update(e) {
        var $target = e.data.target,
            type = e.data.type,
            dataset = e.data.dataset,
            value = $target.val(),
            normalized = options.separator === '.' ? value : value.replace(regex, '.'),
            number = dataset.type === 'int' ? parseInt(normalized, 10) : parseFloat(normalized),
            exponent = Math.pow(10, dataset.digits);

        // Check if the value inside the input field is a number
        if (isNaN(number)) {
            jse.core.debug.info('[NUMBERINPUT] Input is not a number');
            return;
        }

        // Add / substract the stepping value to the value inside the input field
        // If the value gets outside the boundaries set the value to the edge case
        if (type === 'plus') {
            number += dataset.stepping;
        } else {
            number -= dataset.stepping;
        }

        //Check the boundaries given
        number = typeof dataset.max === 'number' ? Math.min(number, dataset.max) : number;
        number = typeof dataset.min === 'number' ? Math.max(number, dataset.min) : number;

        // Round the value to the given digits count
        number = parseInt(Math.round(number * exponent), 10) / exponent;

        // Generate output string
        number = number.toString();

        // Add tailing zeros to get the defined number of digits
        if (dataset.forceDigits) {
            var separatorIndex = number.indexOf('.'),
                digits = null;

            if (separatorIndex === -1) {
                number = number + '.';
                separatorIndex = number.indexOf('.');
            }

            digits = number.length - separatorIndex - 1;

            while (digits < dataset.digits) {
                number += '0';
                digits += 1;
            }
        }

        if ($target.attr('type') !== 'number') {
            // Set the value to the input field in the correct locale
            number = number.replace('.', separator);
        }

        $target.val(number).trigger('keyup', []);

        _quantityCheck($target);
    };

    /**
     * Function to trigger the quantity check
     * @param {object} $target jQuery selector for the input field
     * @private
     */
    var _quantityCheck = function _quantityCheck($target) {
        quantityCheckTimeout = quantityCheckTimeout ? clearTimeout(quantityCheckTimeout) : null;
        quantityCheckTimeout = setTimeout(function () {
            // blur event of input field triggers the CheckStatus request sent in cart_handler widget
            $target.trigger('blur', []);
        }, quantityCheckDelay);
    };

    // ########## EVENT HANDLER ##########

    /**
     * Event handler for the mousedown event. On mousedown
     * on the buttons, the update function gets called after
     * a given delay (that gets shorter after time) as long as
     * no mouseup event is detected
     * @param       {object}    e       jQuery event object
     * @private
     *
     * @todo: search for proper solution to detect a touchend event on integrated android browsers
     */
    var _mouseDown = function _mouseDown(e) {

        e.preventDefault();

        var $target = e.data.target,
            dataset = $target.data(),
            timer = dataset.timer || null,
            delay = Math.max(dataset.delay || e.data.dataset.delay, e.data.dataset.minDelay);

        if (timer) {
            clearTimeout(timer);
        }

        timer = setTimeout(function () {
            _mouseDown(e);
        }, delay);

        $target.data({ delay: delay / 1.5, timer: timer });
        _update(e);
    };

    /**
     * Event handler for the mouseup (and mouseleave) event.
     * If triggered, the timer that gets started in the mouseDown
     * handler gets stopped and all values wil be reseted to the
     * initial state
     *
     * @param       {object}        e       jQuery event object
     * @private
     *
     * @todo: search for proper solution to detect a touchend event on integrated android browsers
     */
    var _mouseUp = function _mouseUp(e) {

        e.preventDefault();

        var $target = e.data ? e.data.target : null,
            dataset = $target !== null ? $target.data() : {},
            timer = dataset.timer;

        if (timer) {
            clearTimeout(timer);
            $target.data('delay', e.data.dataset.delay);
        }
    };

    // ########## INITIALIZATION ##########

    /**
     * Init function of the widget
     * @constructor
     */
    module.init = function (done) {

        separator = _getSeparator();
        regex = new RegExp(separator, 'g');

        $this.find('.input-number').each(function () {
            var $self = $(this),
                $input = $self.find('input'),
                dataset = $.extend({}, options, $self.data());

            $self.on('mousedown', '.btn-plus', {
                dataset: dataset,
                type: 'plus',
                target: $input
            }, _mouseDown).on('mouseup mouseleave', '.btn-plus', {
                dataset: dataset,
                type: 'plus',
                target: $input
            }, _mouseUp).on('mousedown', '.btn-minus', {
                dataset: dataset,
                type: 'minus',
                target: $input
            }, _mouseDown).on('mouseup mouseleave', '.btn-minus', {
                dataset: dataset,
                type: 'minus',
                target: $input
            }, _mouseUp);
        });

        done();
    };

    // Return data to widget engine
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
